module:    Test-Utilities-Implementation
author:    Matthias Hlzl (tc@gauss.muc.de)
copyright: Copyright (C) 1996, 1997, 1998 Matthias Hlzl
           Copyright (C) 1998 Way Forward Technologies
version:   0.03 09 Dec 1998
synopsis:  This Module implements some methods useful for displaying
           the results of running tests.

// Copyright.
// =========

// A library for displaying results in a test-suite.
// Copyright (C) 1998 Matthias Hlzl.
// Copyright (C) 1998 Way Forward Technologies.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston,
// MA 02111-1307  USA
//
// If you need to receive this program under another license contact
// the author (tc@gauss.muc.de).

define variable *nr-of-errors* :: <integer> = 0;
define variable *total-nr-of-errors* :: <integer> = 0;
define variable *errors-found* :: <boolean> = #f;

define variable *silent* :: <boolean> = #f;
define variable *verbose* :: <boolean> = #f;
define variable *say-the-unspeakable* :: <boolean> = #f;

// SILENT-TESTS turns off ANOUNCE and SAY-DONE.
//
define function silent-tests(silent? :: <boolean>)
  if (silent?)
    *silent* := #t;
    *verbose* := #f;
    *say-the-unspeakable* := #f;
  else
    *silent* := #f;
  end if;
end function silent-tests;

// VERBOSE-TESTS turns on MAYBE-SAY.
//
define function verbose-tests(verbose? :: <boolean>)
  if (verbose?)
    *silent* := #f;
    *verbose* := #t;
  else
    *verbose* := #f;
  end if;
end function verbose-tests;

// GABBY-TESTS turns on MAYBE-SAY and DONT-SAY.
//
define function gabby-tests(gabby? :: <boolean>)
  if (gabby?)
    *silent* := #f;
    *verbose* := #t;
    *say-the-unspeakable* := #t;
  else
    *say-the-unspeakable* := #f;
  end if;
end function gabby-tests;

// The method SAY takes a format string and optional arguments
// and formats them on standard output.
//
define method say (#rest arguments)
  let argument-list = concatenate(vector(*standard-output*),
				  arguments);
  if (arguments.size = 1)
    apply(write, argument-list);
  elseif (arguments.size >= 2)
    apply(format, argument-list);
  end if;
  *standard-output*.force-output;
end method say;

// SAY-LINE is used to display a message taking a whole line.
//
define method say-line (string :: <string>)
  write-line(*standard-output*, string);
  *standard-output*.force-output;
end method say-line;

// SAY-NEWLINE is used to display a newline.
//
define method say-newline (#key line-start = "", repeat = 1)
  for (i from 1 to repeat)
    write-line(*standard-output*, "");
  end for;
  say(line-start);
end method say-newline;

define method maybe-say-newline (#key line-start = "", repeat = 1)
  if (*verbose*)
    say-newline(line-start: line-start, repeat: repeat);
  end if;
end method maybe-say-newline;

define method dont-say-newline (#key line-start = "", repeat = 1)
  if (*say-the-unspeakable*)
    say-newline(line-start: line-start, repeat: repeat);
  end if;
end method dont-say-newline;

// private
//
define method say-dots (#rest arguments) => ();
  if (arguments.size > 0)
    apply(do-say-dots, arguments);
  end if;
end method say-dots;

// private
//
define method do-say-dots(string :: <string>, #rest arguments) => ();
  let argument-list
    = concatenate(vector(concatenate(string, "...")),
		  arguments);
  apply(say, argument-list);
end method do-say-dots;

// MAYBE-SAY reports its arguments only if *VERBOSE* is true.
//
define method maybe-say (#rest arguments)
  if (*verbose*)
    apply(say, arguments);
  end if;
end method maybe-say;

// DONT-SAY simply ignores its arguments unless *SAY-THE-UNSPEAKABLE*
// is true.
//
define method dont-say (#rest arguments)
  if (*say-the-unspeakable*)
    apply(say, arguments);
  end if;
end method dont-say;

// ANNOUNCE-SECTION is used to announce a new section of the
// test suite.
//
define method announce-section (string :: <string>, #rest arguments)
  let argument-string
    = apply(format-to-string, string, arguments);
  say(argument-string);
  say-newline();
  for (i from 0 below argument-string.size)
    format(*standard-output*, "=");
  end for;
  say-newline();
end method announce-section;


// ANNOUNCE is used to annouce that a new test starts.
//
define method announce (string :: <string>, #rest arguments)
  unless (*silent*)
    apply(say-dots, string, arguments);
  end unless;
end method announce;

// REPORT-ERROR is the standard way to report errors.
//
define method report-error (#rest arguments) => ();
  *errors-found* := #t;
  *nr-of-errors* := *nr-of-errors* + 1;
  *total-nr-of-errors* := *total-nr-of-errors* + 1;
  unless (arguments.empty?)
    unless (*silent*) say-newline(line-start: "  ") end;
    apply(say-dots, arguments);
  end unless;
end method report-error;

// SAY-DONE is called after each test and produces the correct
// end-of-test message.
//
define method say-done()
  if (*errors-found* = #f)
    unless(*silent*) say-line("done.") end;
  else
    say-line("ERROR!");
    *errors-found* := #f;
  end if;
end method say-done;

// SAY-INTERMEDIATE-RESULTS is used to report the number of
// errors after a part of the test suite.
//
define method say-intermediate-results()
  if (*nr-of-errors* = 0)
  unless(*silent*) say-newline() end;
    say-line("All tests in this section completed without errors.");
  elseif (*nr-of-errors* = 1)
    say-newline();
    say-line("There was one error in this section!");
  else
    say-newline();
    say("There were %D errors in this section!",
	*nr-of-errors*);
  end if;
  say-newline();
  *standard-output*.force-output;
  *nr-of-errors* := 0;
end method say-intermediate-results;

// SAY-RESULTS is used to report a summary of all tests.
//
define method say-results()
  say-newline();
  if (*total-nr-of-errors* = 0)
    say-line("All tests completed without errors.");
  elseif (*total-nr-of-errors* = 1)
    say-line("There was one error in the test-suite!");
  else
    say("There were %D errors in the test-suite!",
	*total-nr-of-errors*);
  end if;
  say-newline(repeat: 2);
  *standard-output*.force-output;
  *nr-of-errors* := 0;
  *total-nr-of-errors* := 0;
end method say-results;
